﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using Inet.Viewer.Data;
using Inet.Viewer.Resources;

namespace Inet.Viewer.WinForms.Prompt
{
    /// <summary>
    /// panel for a range prompt field for which default values can be chosen
    /// </summary>
    public partial class RangePromptFieldWithDefaultValues : PromptControl
    {
        private bool changing, temporaryRangeDefaultInserted;
        private List<PromptValue> defaultValuesRange = new List<PromptValue>();

        /// <summary>
        /// simple constructor with the necessary dependencies
        /// </summary>
        /// <param name="prompt">prompt this panel is based on</param>
        public RangePromptFieldWithDefaultValues(PromptData prompt)
        {
            this.changing = true;
            this.PromptData = prompt;
            InitializeComponent();
            RangePromptValue selectedValue = prompt.Values as RangePromptValue;
            PromptValue fromValueToSelect = null;
            PromptValue toValueToSelect = null;
            PromptValue rangeValueToSelect = null;
            List<PromptValue> defaultValues = new List<PromptValue>();
            List<PromptValue> defaultValuesTo = new List<PromptValue>();
            if (selectedValue != null)
            {
                this.chkIncludeFrom.Checked = ((RangePromptValue)selectedValue).IncludeLow;
                this.chkIncludeTo.Checked = ((RangePromptValue)selectedValue).IncludeHigh;
            }
            else
            {
                this.chkIncludeFrom.Checked = true;
                this.chkIncludeTo.Checked = true;
            }
            ArrayList values = prompt.DefaultValues;

            string selectedRepr = selectedValue == null ? null : selectedValue.StringRepresentation;
            string selectedStartRepr = selectedValue == null ? null : selectedValue.StartValue.StringRepresentation;
            string selectedEndRepr = selectedValue == null ? null : selectedValue.EndValue.StringRepresentation;

            foreach (PromptValue v in values)
            {
                if (v is SinglePromptValue)
                {
                    defaultValues.Add(v);
                    defaultValuesTo.Add(v);
                }
                else if (v is RangePromptValue)
                {
                    defaultValuesRange.Add(v);
                }
                if (selectedValue == null)
                {
                    continue;
                }
                string stringRepr = v.StringRepresentation;
                if (selectedStartRepr == stringRepr)
                {
                    fromValueToSelect = v;
                }
                if (selectedEndRepr == stringRepr)
                {
                    toValueToSelect = v;
                }
                if (selectedRepr == stringRepr)
                {
                    rangeValueToSelect = v;
                }
            }
            if (defaultValues.Count == 0)
            {
                cmbDefaultValues.DropDownStyle = ComboBoxStyle.Simple;
            }
            else
            {
                this.cmbDefaultValues.DataSource = defaultValues;
                this.cmbDefaultValues.DisplayMember = "DisplayString";
                this.cmbDefaultValues.ValueMember = "Value";
                AutoSizeComboBox(cmbDefaultValues);
            }

            if (defaultValuesTo.Count == 0)
            {
                cmbDefaultValuesTo.DropDownStyle = ComboBoxStyle.Simple;
            }
            else
            {
                this.cmbDefaultValuesTo.DataSource = defaultValuesTo;
                this.cmbDefaultValuesTo.DisplayMember = "DisplayString";
                this.cmbDefaultValuesTo.ValueMember = "Value";
                AutoSizeComboBox(cmbDefaultValuesTo);
            }

            if (defaultValuesRange.Count == 0)
            {
                cmbRangeDefaults.Visible = false;
            }
            else
            {
                this.cmbRangeDefaults.Items.AddRange(defaultValuesRange.ToArray());
                this.cmbRangeDefaults.DisplayMember = "DisplayString";
                this.cmbRangeDefaults.ValueMember = "Value";
            }


            if (selectedValue != null)
            {
                this.chkIncludeFrom.Checked = selectedValue.IncludeLow;
                this.chkIncludeTo.Checked = selectedValue.IncludeHigh;
            }


            if (selectedValue == null && this.cmbRangeDefaults.Items.Count > 0)
            {
                selectedValue = this.cmbRangeDefaults.Items[0] as RangePromptValue;
            }
            if (fromValueToSelect != null)
            {
                this.cmbDefaultValues.SelectedItem = fromValueToSelect;
            }
            else if (selectedValue != null)
            {
                this.cmbDefaultValues.SelectedItem = null;
                this.cmbDefaultValues.Text = selectedValue.StartValue.ValueString;
            }
            if (toValueToSelect != null)
            {
                this.cmbDefaultValuesTo.SelectedItem = toValueToSelect;
            }
            else if (selectedValue != null)
            {
                this.cmbDefaultValuesTo.SelectedItem = null;
                this.cmbDefaultValuesTo.Text = selectedValue.EndValue.ValueString;
            }
            if (cmbRangeDefaults.Visible)
            {
                if (rangeValueToSelect != null)
                {
                    this.cmbRangeDefaults.SelectedItem = rangeValueToSelect;
                }
                else if (selectedValue != null)
                {
                    UpdateRangeFromSingle();
                }
            }

            if (!this.PromptData.Changeable)
            {
                this.cmbDefaultValues.DropDownStyle = ComboBoxStyle.DropDownList;
                this.cmbDefaultValuesTo.DropDownStyle = ComboBoxStyle.DropDownList;
                this.cmbRangeDefaults.DropDownStyle = ComboBoxStyle.DropDownList;
            }
            changing = false;
            ControlForErrors = chkIncludeTo;
            ErrorProvider = errorProvider;
        }

        /// <summary>
        /// <inheritDoc/>
        /// </summary>
        internal override PromptValue Value
        {
            get
            {
                try
                {
                    object fromItem = this.cmbDefaultValues.SelectedItem;
                    SinglePromptValue start;
                    if (fromItem is SinglePromptValue)
                    {
                        start = fromItem as SinglePromptValue;
                    }
                    else
                    {
                        start = new SinglePromptValue(PromptData.Type, this.cmbDefaultValues.Text);
                    }
                    SinglePromptValue end;
                    object toItem = this.cmbDefaultValuesTo.SelectedItem;
                    if (toItem is SinglePromptValue)
                    {
                        end = toItem as SinglePromptValue;
                    }
                    else
                    {
                        end = new SinglePromptValue(PromptData.Type, this.cmbDefaultValuesTo.Text);
                    }
                    return new RangePromptValue(start, end, chkIncludeFrom.Checked, chkIncludeTo.Checked, PromptData.Type);
                }
                catch (Exception)
                {
                    return null;
                }
            }
        }
 
        /// <summary>
        /// notifies that the selected value has changed
        /// </summary>
        /// <param name="sender">object responsible for the event</param>
        /// <param name="e">eventargs of the action</param>
        private void cmbDefaultValues_SelectedValueChanged(object sender, EventArgs e)
        {
            if (changing)
            {
                return;
            }
            changing = true;
            try
            {
                if (sender == this.cmbRangeDefaults)
                {
                    RangePromptValue range = this.cmbRangeDefaults.SelectedItem as RangePromptValue;
                    this.cmbDefaultValues.SelectedItem = null;
                    this.cmbDefaultValuesTo.SelectedItem = null;
                    if (range != null)
                    {
                        this.cmbDefaultValues.Text = range.StartValue.DisplayString;
                        this.cmbDefaultValuesTo.Text = range.EndValue.DisplayString;
                        this.chkIncludeFrom.Checked = range.IncludeLow;
                        this.chkIncludeTo.Checked = range.IncludeHigh;
                    }
                    if (temporaryRangeDefaultInserted && cmbRangeDefaults.SelectedIndex > 0)
                    {
                        cmbRangeDefaults.Items.RemoveAt(0);
                        temporaryRangeDefaultInserted = false;
                    }
                }
                else
                {
                    UpdateRangeFromSingle();
                }
                ValidatePrompt();
                OnValueChanged();
            }
            finally
            {
                changing = false;
            }
        }

        /// <summary>
        /// Updates the range value from the single values. A temporary entry 
        /// will be added to combobox when the entered range does not exist
        /// as default range.
        /// </summary>
        private void UpdateRangeFromSingle()
        {
            RangePromptValue range = this.Value as RangePromptValue;
            if (range != null)
            {
                PromptValue defaultValue = null;
                string rangeStrRep = range.StringRepresentation;
                foreach(PromptValue value in defaultValuesRange) {
                    if (value.StringRepresentation == rangeStrRep) {
                        defaultValue = value;
                    }
                }
                if (defaultValue != null) {
                    if (temporaryRangeDefaultInserted)
                    {
                        cmbRangeDefaults.Items.RemoveAt(0);
                        temporaryRangeDefaultInserted = false;
                    }
                    cmbRangeDefaults.SelectedItem = defaultValue;
                } else {
                    if (temporaryRangeDefaultInserted)
                    {
                        cmbRangeDefaults.Items[0] = range;
                    }
                    else
                    {
                        cmbRangeDefaults.Items.Insert(0, range);
                        temporaryRangeDefaultInserted = true;
                    }
                    cmbRangeDefaults.SelectedIndex = 0;
                }
            }
        }

        /// <summary>
        /// notifies a change in the "null" checkbox state
        /// </summary>
        /// <param name="sender">object responsible for the event</param>
        /// <param name="e">eventargs of the action</param>
        private void chk_CheckedChanged(object sender, EventArgs e)
        {
            if (changing)
            {
                return;
            }
            changing = true;
            UpdateRangeFromSingle();
            changing = false;
        }

        /// <inheritdoc/>
        internal override bool ValidatePrompt()
        {
            bool r = true;
            errorProvider.Clear();
            try
            {
                r &= cmbDefaultValues.SelectedItem != null || ValidatePrompt(new SinglePromptValue(PromptData.Type, cmbDefaultValues.Text), chkIncludeFrom);
            }
            catch (Exception e)
            {
                errorProvider.SetError(chkIncludeFrom, e.Message);
                ErrorMessage = e.Message;
                r = false;
            }
            try
            {
                if (cmbDefaultValuesTo.SelectedItem == null && !ValidatePrompt(new SinglePromptValue(PromptData.Type, cmbDefaultValuesTo.Text), chkIncludeTo))
                {
                    r = false;
                }
            }
            catch (Exception e)
            {
                errorProvider.SetError(chkIncludeTo, e.Message);
                ErrorMessage = e.Message;
                r = false;
            }
            return r && ValidatePrompt(Value, ControlForErrors);
        }
    }
}
